<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/ /** */

require_once 'database_model.php'; //not using the loader because we don't want to instantiate the class
require_once APPPATH.'models/message.php';

/**
* Parent class for models that need to track a custom flag on a message.
*
* Custom flags consist of three parts:
* - A flag that indicates whether or not the custom flag should be considered active
* - A flag that indicates a database record that contains metadata for the flag.  This will be preserved even when the flag is inactive.
* - A database record with metadata for the flag. 
*
* In order to reduce the number of API calls needed, the data for the message will be encapsulated in a {@link Message} object, to make it easier
* to preserve information pulled from the API even if the message is passed between multiple methods.  Most of the methods in this model will take
* either the message id or a message object as a parameter.  If you already have a message object instantiated, it will reduce API overhead to 
* pass that instead of the id.
*
* @package direct-project-innovation-initiative
* @subpackage models
*/
class API_custom_flag_model extends Database_Model {	

	function __construct(){
		parent::__construct();
	}
		
	/**
	* Finds the database entry that belongs to the given message.
	*
	* Note that if you know the database entry's id or another way to look it up in the db, it's more efficient
	* to use {@link find()} or {@link find_one()}.  This will connect to the API and check the header to determine
	* the id for the database entry before looking it up.
	*
	* @param Message|scalar 
	* @return array|false
	*/
	public function find_for_message($message_uid_or_object){		
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		return $this->find_one(array('message_id'=>$message->id));
	}
	
	public function find_for_messages($message_ids){
		if(!$this->is->array_of_nonzero_unsigned_integers($message_ids)) return $this->error->should_be_an_array_of_nonzero_unsigned_integers();
		$this->db->where_in('message_id', $message_ids);
		return $this->find();
	}
	
	
	/**
	* True if the custom flag is active on this message.
	*
	* Use {@link exists_for_message} to determine if there is a database record for a custom flag on this message.
	* Database records are preserved for de-activated flags so that they can be reused if the flag is re-actiivated.
	*
	* @param Message|scalar
	* @return boolean
	*/
	public function active_for_message($message_uid_or_object){
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		$active = $this->find_for_message($message)['active'];
		return ($active && $this->exists_for_message($message));	
	}
	
	/**
	* True if there is a database entry for the custom flag on this message.
	*
	* Use {@link active_for_message} to determine if the custom flag is currently active; database records are preserved when the
	* flag is deactivated so that they may be re-used later if the flag is re-activated.
	*
	* @param Message|scalar
	* @return boolean
	*/
	public function exists_for_message($message_uid_or_object){
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		return isset($this->find_for_message($message)['id']);
	}

	/**
	* Sets the custom flag for this message.
	*
	* This will will activate the custom flag and create or edit the database record for it as needed.  
	* To deactivate the flag and clear its metadata, use {@link remove_from_message}.
	*
	* Note that this is no longer a significantly greater overhead than using create_for_message or update_for_message.
	*
	* @param Message|scalar
	* @param array
	* @return array|false
	*/
	public function set_for_message($message_uid_or_object, $values){
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages		
		if(!$this->is->nonempty_array($values)) return $this->error->should_be_a_nonempty_array($values);
		
		if($this->exists_for_message($message)){
			return $this->update_for_message($message, $values);
		}
		
		return $this->create_for_message($message, $values);	
	}

	/**
	* Creates a database record for {@link table_name} & associates it with the given message.
	*
	* If you are unsure whether or not a message is flagged, please use {@link set_for_message()} instead - 
	* this will first check to see whether a flag already exists before adding a row to the database.
	*
	* @param Message|scalar
	* @param array
	* @return array|false
	*/
	function create_for_message($message_uid_or_object, $values) {
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		
		$values['message_id'] = $message->id;
		
		//add a record for this custom flag to the database
		$record = $this->create($values);
		if(empty($record)) return false; //create() failed and will have triggered relevant errors
		
		return $record;
	}
	
	/**
	* Updates the database record of the custom flag for this message.
	*
	* Supplying the $record_id is optional, but leaving it out will slightly increase the overhead of this method.
	*
	* @param Message|scalar
	* @param array
	* @param int 
	* @return array
	*/
	function update_for_message($message_uid_or_object, $values, $record_id = null){
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		if(!$this->is->nonempty_array($values)) return $this->error->should_be_a_nonempty_array($values);
		if(!is_null($record_id) && !$this->formatted_like_an_id($record_id)) return $this->error->should_be_an_x($this->model_alias.' id ', $record_id);
		$flag_data = $this->find_for_message($message);
		//make sure that we have the record id for this custom flag
		if(is_null($record_id)){
			$record_id = $flag_data['id'];
			if(!$this->formatted_like_an_id($record_id)) return $this->error->should_be_an_x('the '.$this->model_alias.' id for '.$message->id_for_error(), $record_id);	
		}
		
		$active_at_start = element('active', $flag_data);
		if(!$active_at_start){
			//since we're re-activating a de-activated flag, we'll want to replace the existing created_at timestamp with now
			$now = now();
			$values['created_at'] = $now;
			$values['modified_at'] = $now; 
		}
		
		$record = $this->update($record_id, $values);
		if(empty($record)){
			if(!$active_at_start) {
				//update to inactive
			}
			return false; //update() will trigger any necessary errors			
		}
		return $record;
	}
	
	/**
	* Deactivates the custom flag on this memory and clears its metadata from the database.
	*
	* Note that the database record for this custom flag will still exist in the database and its id will remain associated
	* with the message.  The same database record will be used again if this custom flag is reactivated.
	*
	* @param Message|scalar
	* @return boolean
	*/
	function remove_from_message($message_uid_or_object){
		$message = get_message($message_uid_or_object);
		if(!is_a($message, 'Message')) return false; //get_message() will trigger any necessary error messages
		
		//if the db update fails, we'll be ok - we'll overwrite the db values the next time that we flag the message
		$record_id = $this->find_for_message($message)['id'];
		$this->_clear_record($record_id);
		
		return true;
	}
	
	/**
	* True if the given value is formatted like the unique API id for a message.
	* @param mixed
	* @return boolean
	*/
	function formatted_like_a_message_id($message_uid){ 
		return is_scalar($message_uid) && !empty($message_uid);
	}
	
	protected function _clear_record($record_id){ $this->error->notice('Please define '.get_class($this).'::_clear_flag_metadata()'); }
		
}
